package jcircus.gui;

import circparser.formw.util.DeclName;
import java.awt.ComponentOrientation;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import jcircus.exceptions.FailParsingException;
import jcircus.exceptions.FailTypeCheckingException;
import jcircus.exceptions.InvalidParameterException;
import jcircus.util.ProcInfo;
import jcircus.util.Util;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.VarDecl;

/**
 * ParametersDialog.java
 *
 * Dialog where the user enters the parameters for a parametrised process.
 *
 * @author Angela
 */
public class ParametersDialog extends JDialog implements ActionListener {

    private ProcInfo procInfo;
    private JTextField inputFields[];
    private JButton btnOk;
    private JButton btnCancel;
    private JCircusFrame jCircusFrame;

    /**
     * Constructor.
     */
    public ParametersDialog(JCircusFrame jCircusFrame, ProcInfo procInfo) {
        super(jCircusFrame, "Parameters", true);
        
        this.jCircusFrame = jCircusFrame;
        this.procInfo = procInfo;
        
        // User can not close by clicking on "X"
        this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        this.getContentPane().setLayout(
                new BoxLayout(this.getContentPane(), BoxLayout.Y_AXIS));
        
        String[] text = new String[] {
            "Process " + procInfo.getProcessName() + " requires parameters.",
            "Please enter their values."
        };
        
        for (int i = 0; i < text.length; i++) {
            this.getContentPane().add(new JLabel(text[i]));
        }
        
        List<VarDecl> params = procInfo.getParameters();
        
        JPanel pnlParams = new JPanel();
        pnlParams.setLayout(new GridLayout((params.size() + 1), 3));
        pnlParams.setComponentOrientation(ComponentOrientation.LEFT_TO_RIGHT);
        
        pnlParams.add(new JLabel("Parameter"));
        pnlParams.add(new JLabel("Type"));
        pnlParams.add(new JLabel("Value"));
        
        inputFields = new JTextField[params.size()];
        for (int i = 0; i < params.size(); i++) {
            // VarDecl declares at most one variable. Gets this variable.
            String param = params.get(i).getDeclName().get(0).toString();
            // Gets the expression in the VarDecl.
            Expr expr = params.get(i).getExpr();
            
            String type;
            if (expr instanceof RefExpr) {
                // The type expression should be a name, that is, a reference to 
                // a Free Type or to \nat or \num or \arithmos. In the future, 
                // other types will be considered as well.
                type = (String) ((RefExpr) expr).getRefName().toString();
            } else {
                throw new InvalidParameterException("");
            }
            pnlParams.add(new JLabel(param));
            pnlParams.add(new JLabel(type));
            inputFields[i] = new JTextField();
            pnlParams.add(inputFields[i]);
        }
        
        this.getContentPane().add(pnlParams);
        
        JPanel pnlButtons = new JPanel();
        btnOk = new JButton("OK");
        btnOk.setSize(20, 10);
        btnOk.addActionListener(this);
        pnlButtons.add(btnOk);

        btnCancel = new JButton("Cancel");
        btnCancel.setSize(20, 10);
        btnCancel.addActionListener(this);
        pnlButtons.add(btnCancel);
        
        this.getContentPane().add(pnlButtons);
    }
    
    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
    
        if (e.getSource() == btnOk) {
            btnOkPressed();
        } else if (e.getSource() == btnCancel) {
            btnCancelPressed();
        }
    }
    
    /**
     * Code for when button OK is pressed.
     *
     */
    private void btnOkPressed() {
    
        List<VarDecl> params = procInfo.getParameters();
        Expr expr;

        // Tests if all the parameters have been entered
        for (int i = 0; i < inputFields.length; i++) {
            if (inputFields[i].getText().equals("")) {
                JOptionPane.showMessageDialog(this, 
                        "You must enter all the parameters.");
                return;
            }
        }
        
        for (int i = 0; i < inputFields.length; i++) {
            
            // Gets the expression in the VarDecl.
            String actualSt = inputFields[i].getText().trim();
            Expr typeExpr = ((VarDecl) params.get(i)).getExpr();

            try {
                String codeForActual = jCircusFrame.getJCircus().
                        getTranslator2Java().validateAndReturnCodeForParameter(
                            actualSt, typeExpr);

                procInfo.getCodeForActuals().add(i, codeForActual);

            } catch(FailParsingException fpe) {
                JOptionPane.showMessageDialog(this, fpe.getFirstError(),
                        "Parsing Error", JOptionPane.ERROR_MESSAGE);
                inputFields[i].selectAll();

            } catch(FailTypeCheckingException ftce) {
                JOptionPane.showMessageDialog(this, ftce.getFirstError(),
                        "Type Checking Error", JOptionPane.ERROR_MESSAGE);
                inputFields[i].selectAll();
            }
        }

        // Finishes and closes the window
        ((JCircusFrame) this.getOwner()).setCanceled(false);
        this.setVisible(false);
        this.dispose();
        
    }
    
    /**
     * Code for when button Cancel is pressed.
     *
     */    
    private void btnCancelPressed() {

        // Finishes and closes the window
        ((JCircusFrame) this.getOwner()).setCanceled(true);
        this.setVisible(false);
        this.dispose();
    }    
}
